import { Currency, CurrencyAmount, Percent, Price, TradeType } from '@uniswap/sdk-core';
import { Pair, Route as V2RouteSDK } from '@uniswap/v2-sdk';
import { Pool as V3Pool, Route as V3RouteSDK } from '@uniswap/v3-sdk';
import { Pool as V4Pool, Route as V4RouteSDK } from '@uniswap/v4-sdk';
import { MixedRouteSDK } from './mixedRoute/route';
import { IRoute } from './route';
export declare class Trade<TInput extends Currency, TOutput extends Currency, TTradeType extends TradeType> {
    readonly routes: IRoute<TInput, TOutput, Pair | V3Pool | V4Pool>[];
    readonly tradeType: TTradeType;
    private _outputAmount;
    private _inputAmount;
    private _nativeInputRoutes;
    private _wethInputRoutes;
    /**
     * The swaps of the trade, i.e. which routes and how much is swapped in each that
     * make up the trade. May consist of swaps in v2 or v3.
     */
    readonly swaps: {
        route: IRoute<TInput, TOutput, Pair | V3Pool | V4Pool>;
        inputAmount: CurrencyAmount<TInput>;
        outputAmount: CurrencyAmount<TOutput>;
    }[];
    constructor({ v2Routes, v3Routes, v4Routes, mixedRoutes, tradeType, }: {
        v2Routes?: {
            routev2: V2RouteSDK<TInput, TOutput>;
            inputAmount: CurrencyAmount<TInput>;
            outputAmount: CurrencyAmount<TOutput>;
        }[];
        v3Routes?: {
            routev3: V3RouteSDK<TInput, TOutput>;
            inputAmount: CurrencyAmount<TInput>;
            outputAmount: CurrencyAmount<TOutput>;
        }[];
        v4Routes?: {
            routev4: V4RouteSDK<TInput, TOutput>;
            inputAmount: CurrencyAmount<TInput>;
            outputAmount: CurrencyAmount<TOutput>;
        }[];
        mixedRoutes?: {
            mixedRoute: MixedRouteSDK<TInput, TOutput>;
            inputAmount: CurrencyAmount<TInput>;
            outputAmount: CurrencyAmount<TOutput>;
        }[];
        tradeType: TTradeType;
    });
    get inputAmount(): CurrencyAmount<TInput>;
    get outputAmount(): CurrencyAmount<TOutput>;
    /**
     * Returns the sum of all swaps within the trade
     * @returns
     * inputAmount: total input amount
     * inputAmountNative: total amount of native currency required for ETH input paths
     *  - 0 if inputAmount is native but no native input paths
     *  - undefined if inputAmount is not native
     * outputAmount: total output amount
     * outputAmountNative: total amount of native currency returned from ETH output paths
     *  - 0 if outputAmount is native but no native output paths
     *  - undefined if outputAmount is not native
     */
    get amounts(): {
        inputAmount: CurrencyAmount<TInput>;
        inputAmountNative: CurrencyAmount<TInput> | undefined;
        outputAmount: CurrencyAmount<TOutput>;
        outputAmountNative: CurrencyAmount<TOutput> | undefined;
    };
    get numberOfInputWraps(): number;
    get numberOfInputUnwraps(): number;
    get nativeInputRoutes(): IRoute<TInput, TOutput, Pair | V3Pool | V4Pool>[];
    get wethInputRoutes(): IRoute<TInput, TOutput, Pair | V3Pool | V4Pool>[];
    private _executionPrice;
    /**
     * The price expressed in terms of output amount/input amount.
     */
    get executionPrice(): Price<TInput, TOutput>;
    /**
     * Returns the sell tax of the input token
     */
    get inputTax(): Percent;
    /**
     * Returns the buy tax of the output token
     */
    get outputTax(): Percent;
    private isWrappedNative;
    /**
     * The cached result of the price impact computation
     * @private
     */
    private _priceImpact;
    /**
     * Returns the percent difference between the route's mid price and the expected execution price
     * In order to exclude token taxes from the price impact calculation, the spot price is calculated
     * using a ratio of values that go into the pools, which are the post-tax input amount and pre-tax output amount.
     */
    get priceImpact(): Percent;
    /**
     * Get the minimum amount that must be received from this trade for the given slippage tolerance
     * @param slippageTolerance The tolerance of unfavorable slippage from the execution price of this trade
     * @returns The amount out
     */
    minimumAmountOut(slippageTolerance: Percent, amountOut?: CurrencyAmount<TOutput>): CurrencyAmount<TOutput>;
    /**
     * Get the maximum amount in that can be spent via this trade for the given slippage tolerance
     * @param slippageTolerance The tolerance of unfavorable slippage from the execution price of this trade
     * @returns The amount in
     */
    maximumAmountIn(slippageTolerance: Percent, amountIn?: CurrencyAmount<TInput>): CurrencyAmount<TInput>;
    /**
     * Return the execution price after accounting for slippage tolerance
     * @param slippageTolerance the allowed tolerated slippage
     * @returns The execution price
     */
    worstExecutionPrice(slippageTolerance: Percent): Price<TInput, TOutput>;
    static fromRoutes<TInput extends Currency, TOutput extends Currency, TTradeType extends TradeType>(v2Routes: {
        routev2: V2RouteSDK<TInput, TOutput>;
        amount: TTradeType extends TradeType.EXACT_INPUT ? CurrencyAmount<TInput> : CurrencyAmount<TOutput>;
    }[], v3Routes: {
        routev3: V3RouteSDK<TInput, TOutput>;
        amount: TTradeType extends TradeType.EXACT_INPUT ? CurrencyAmount<TInput> : CurrencyAmount<TOutput>;
    }[], tradeType: TTradeType, mixedRoutes?: {
        mixedRoute: MixedRouteSDK<TInput, TOutput>;
        amount: TTradeType extends TradeType.EXACT_INPUT ? CurrencyAmount<TInput> : CurrencyAmount<TOutput>;
    }[], v4Routes?: {
        routev4: V4RouteSDK<TInput, TOutput>;
        amount: TTradeType extends TradeType.EXACT_INPUT ? CurrencyAmount<TInput> : CurrencyAmount<TOutput>;
    }[]): Promise<Trade<TInput, TOutput, TTradeType>>;
    static fromRoute<TInput extends Currency, TOutput extends Currency, TTradeType extends TradeType>(route: V2RouteSDK<TInput, TOutput> | V3RouteSDK<TInput, TOutput> | V4RouteSDK<TInput, TOutput> | MixedRouteSDK<TInput, TOutput>, amount: TTradeType extends TradeType.EXACT_INPUT ? CurrencyAmount<TInput> : CurrencyAmount<TOutput>, tradeType: TTradeType): Promise<Trade<TInput, TOutput, TTradeType>>;
}
